/////////////////////////////////////////////////////////////////////
//
// PhilipsHue.h Library, (c) 2015, Ryan A. Colyer
// Distributed under the Boost Software License, v1.0. (LICENSE.txt)
//
// (Philips Hue is a trademark of Philips.  This software is
// unaffiliated with Philips.)
//
/////////////////////////////////////////////////////////////////////

#include "http.h"
#include "RC/RC.h"

// A class for controlling the Philips Hue bulbs, following the quick-start
// guide here:  http://www.developers.meethue.com/documentation/getting-started
class PhilipsHue {
  protected:

  RC::RStr address;
  RC::RStr username;

  void frange(f32 &x) const {
    if (x < 0) { x = 0; }
    if (x > 1) { x = 1; }
  }

  public:

  // Set the address (ip address, most likely), and the username given
  // permission.
  PhilipsHue(RC::RStr address, RC::RStr username)
    : address(address),
      username(username) {
  }

  // Sets a bulb's brightness (0-255), saturation (0-255) and hue (0-65535).
  // If brightness is 0, the bulb is turned off.
  void BulbBSH(u32 bulb, u32 bri_int, u32 sat_int, u32 hue_int) const {
    RC::RStr onstr = (bri_int == 0) ? "false" : "true";

    RC::RStr response;

    try {
      response = HTTP::Put("http://" + address + "/api/" + username
                + "/lights/" + RC::RStr(bulb) + "/state",
                "{\"on\":" + onstr + ", \"sat\":" + RC::RStr(sat_int)
                + ", \"bri\":" + RC::RStr(bri_int) + ", \"hue\":"
                + RC::RStr(hue_int) + "}");
    }
    catch (RC::ErrorMsg &e) {
      std::cerr << e.GetError() << std::endl;
    }


    if (response.find("[{\"error\":") != response.npos) {
      std::cerr << response << std::endl;
    }
  }

  // Sets a bulb by RGB values (0-1.0 each).  If all three are 0, the bulb
  // is turned off.
  void Bulb(u32 bulb, f32 r, f32 g, f32 b) const {
    frange(r);  frange(g);  frange(b);

    // RGB to brightness, saturation, and hue.
    f32 hue_x = r + g*cos(2.094395102393193) + b*cos(4.188790204786387);
    f32 hue_y = g*sin(2.094395102393193) + b*sin(4.188790204786387);
    f32 hue = std::atan2(hue_y, hue_x);
    if (hue < 0) { hue += 6.283185307179586; }
    f32 max = std::max({r, g, b});
    f32 min = std::min({r, g, b});
    f32 bri = max;
    f32 sat = (max - min)/max;

    u32 bri_int = bri * 255.999;
    u32 sat_int = sat * 255.999;
    u32 hue_int = hue * 65535.99 / 6.283185307179586;

    BulbBSH(bulb, bri_int, sat_int, hue_int);
  }

  // Sets a bulb's brightness to val (0-1.0), using white light.
  // If val is 0, it is turned off.
  void Bulb(u32 bulb, f32 val) const {
    Bulb(bulb, val, val, val);
  }


  // Registers the username.  Returns error message or "" if succeeded.
  RC::RStr Link() const {
    RC::RStr response;

    try {
      response = HTTP::Post("http://" + address + "/api",
                "{\"devicetype\":\"" + username + "\",\"username\":\""
                + username + "\"}");
    }
    catch (RC::ErrorMsg &e) {
      std::cerr << e.GetError() << std::endl;
    }

    if (response.find("[{\"success\":") != response.npos) {
      return "";
    }
    else {
      return response;
    }
  }
  

};

