/////////////////////////////////////////////////////////////////////
//
// RC Library, (c) 2011-2014, Ryan A. Colyer
// Distributed under the Boost Software License, v1.0. (LICENSE.txt)
//
/// \file RCBits.h
/// Provides short convenience classes and functions.
/////////////////////////////////////////////////////////////////////

#ifndef RC_RCBITS_H
#define RC_RCBITS_H

#include "RCconfig.h"
#ifdef CPP11
#include "Tuple.h"
#endif

namespace RC {

/// @cond UNDOC
template<class T>
class BetwCompare {
  protected:
  const T& x;
  bool ans;
  public:
  inline explicit BetwCompare(const T& x) : x(x), ans(true) { }
#define RC_BETW_HELP(REL) \
  template<class T2> \
  inline BetwCompare operator REL (const T2& other) { \
    ans &= x REL other; \
    return *this; \
  }
  RC_BETW_HELP(<)
  RC_BETW_HELP(>)
  RC_BETW_HELP(<=)
  RC_BETW_HELP(>=)
  RC_BETW_HELP(==)
  RC_BETW_HELP(!=)
  inline operator bool () const { return ans; }
};

#define RC_BETW_HELP2(REL1, REL2) \
template<class T, class T2> \
inline BetwCompare<T> operator REL1 (const T2& other, BetwCompare<T> betw) { \
  return betw REL2 other; \
}

RC_BETW_HELP2(<, >)
RC_BETW_HELP2(>, <)
RC_BETW_HELP2(<=, >=)
RC_BETW_HELP2(>=, <=)
RC_BETW_HELP2(==, ==)
RC_BETW_HELP2(!=, !=)

/// @endcond

/// Returns a comparator that can be used for range comparisons.
/** Usage exmple:  if (3.14 < Betw(5) <= 7) { cout << "Yes\n"; }
 *  Works for any comparable types, as long as one of the left-most two in
 *  the chain is wrapped in Betw.
 */
template<class T>
BetwCompare<T> Betw(const T& x) { return BetwCompare<T>(x); }

#ifdef CPP11

/// @cond UNDOC
template<class... Args>
class OneOfCompare {
  protected:
  Tuple<Args...> compare;
  public:
  OneOfCompare(Tuple<Args...>&& tup) : compare(tup) { }
  template<class T, class... Args2>
  friend bool operator==(const T& item, const OneOfCompare<Args2...>& list);
  template<class T, class... Args2>
  friend bool operator!=(const T& item, const OneOfCompare<Args2...>& list);
};

template<class T, class... Args>
bool operator== (const T& item, const OneOfCompare<Args...>& list) {
  Data1D<T> arr = list.compare.template AsData<T>();
  for (T& elem : arr) {
    if (item == elem) {
      return true;
    }
  }
  return false;
}
template<class T, class... Args>
bool operator!= (const T& item, const OneOfCompare<Args...>& list) {
  return ! (item == list);
}
/// @endcond

/// Returns a comparator that can be used to check if something is in a list.
/** Each element given as a parameter to OneOf is constructed with the type
 *  of the object left of the == or != operator.  Usage exmple:
 *  RStr str = "foo";  if (str != OneOf("bar", -3.54, "blah")) {
 *  cout << "Not Found\n"; }  int x = 8; if (5 == OneOf(3.14, 5, x)) {
 *  cout << "Found\n"; }
 */
template<class... Args>
OneOfCompare<Args...> OneOf(Args... args) {
  return OneOfCompare<Args...>(Tuple<Args...>(args...));
}
#endif


/// A size_t like integer class which automatically stays within its range.
/** Use this for circular buffers or anywhere modulo arithmetic is needed.
 */
class LoopIndex {
  public:
  /// Defines the range of of the object.  The index is always less than this.
  inline LoopIndex(size_t range) : index(0), range(range) { }
  /// Returns the set range.
  inline size_t Range() const { return range; }
  /// Sets a new range.
  inline void SetRange(size_t new_range) { range = new_range; *this = index; }
  /// Assigns a new index value, forcing it in the range.  Signed indices are
  /// handled properly.
  inline LoopIndex& operator= (size_t new_index) {
    if (new_index > ((size_t)(-1)/2)) {
      index = range - ((0-new_index) % range);
    }
    else {
      index = new_index % range;
    }
    return *this;
  }
  /// Implicitly returns the size_t index.
  inline operator size_t () const { return index; }
  /// Increment by one, looping within range.
  inline LoopIndex& operator++ () {
    index++;
    if (index>=range) { index=0; }
    return *this;
  }
  /// Postfix increment by one, looping within range.
  inline LoopIndex operator++ (int) {
    LoopIndex tmp(*this);
    ++(*this);
    return tmp;
  }
  /// Decrement by one, looping within range.
  inline LoopIndex& operator-- () {
    if (index==0) { index = range-1; }
    else { index--; }
    return *this;
  }
  /// Postfix decrement by one, looping within range.
  inline LoopIndex operator-- (int) {
    LoopIndex tmp(*this);
    --(*this);
    return tmp;
  }
  /// Increment by offset, looping within range, and handling negative offsets
  /// correctly.
  inline LoopIndex& operator+= (size_t offset) {
    if (offset > ((size_t)(-1)/2)) {
      size_t negoff = (0-offset)%range;
      if (index < negoff) {
        index = range - (negoff - index);
      }
      else {
        index = index - negoff;
      }
    }
    else {
      index = (index + offset) % range;
    }
    return *this;
  }
  /// Decrement by offset, looping within range, and handling negative offsets
  /// correctly.
  inline LoopIndex& operator-= (size_t offset) {
    return (*this += 0-offset);
  }

  protected:
  /// @cond PROTECTED
  size_t index;
  size_t range;
  /// @endcond
};

}

#endif // RC_RCBITS_H

