/////////////////////////////////////////////////////////////////////
//
// hueset.cpp, (c) 2015, Ryan A. Colyer
// Distributed under the Boost Software License, v1.0. (LICENSE.txt)
//
// (Philips Hue is a trademark of Philips.  This software is
// unaffiliated with Philips.)
//
// This program is a simple demonstration of using the
// Philips Hue http API in C++.  It allows setting the brightness or
// the RGB values.  It also demonstrates two light shows, one cycling
// through the spectrum, and the other emulating Christmas lights.
//
/////////////////////////////////////////////////////////////////////

#include "PhilipsHue.h"
#include "RC/RC.h"

RC_USE

#define HelpExit() \
  cerr << "Manually set the color (all values range from 0 to 1.0)\n" \
       << "  " << args[0] << " <brightness>\n" \
       << "  " << args[0] << " <red> <green> <blue>\n" \
       << "\nLight show modes:\n" \
       << "  " << args[0] << " spectrum\n" \
       << "  " << args[0] << " christmas\n" \
       << "\nTo register with a new bridge controller:\n" \
       << "  " << args[0] << " link\n"; \
  return -1;


// Cycles through the spectrum repeatedly.
void Spectrum(PhilipsHue& hue, Data1D<u32> bulb_set) {
  u16 val = 0;
  while (1) {
    for(u32 bulb: bulb_set) {
      hue.BulbBSH(bulb, 255, 255, val);
      Time::Sleep(0.1);
    }
    val+= 2000;
  }
}


// Randomly switches lights between red, green, and blue.
void Christmas(PhilipsHue& hue, Data1D<u32> bulb_set) {
  RND rng;
  while (1) {
    for(u32 bulb: bulb_set) {
      u32 rval = rng.GetRange(3);
      switch (rval) {
        case 0:  hue.Bulb(bulb, 1, 0, 0);  break;
        case 1:  hue.Bulb(bulb, 0, 1, 0);  break;
        case 2:  hue.Bulb(bulb, 0, 0, 1);  break;
        default:  Throw_RC_Error("RNG error");  break;
      }
      Time::Sleep(3.0);
    }
  }
}


// Register the username.
void Link(PhilipsHue &hue) {
  std::cout << "Press the 'link' button in the middle of your controller,\n"
       << "then press enter here.\n";
  std::cin.get();

  RStr response = hue.Link();
  if (response == "") {
    std::cout << "Username registered.\n";
  }
  else {
    std::cerr << "Error registering username:\n";
    std::cerr << response << std::endl;
  }
}


RC_MAIN {
  if (args.size() != OneOf(2, 4)) {
    HelpExit();
  }

  // Set the ip address and the username.
  PhilipsHue hue("192.168.0.50", "huesetuser");
  // Initialize the list of bulbs to control.
  Data1D<u32> bulb_set{1, 2, 3};

  if (args.size() == 2 && (! args[1].Is_f32())) {
    if (args[1] == "spectrum") {
      Spectrum(hue, bulb_set);
    }
    else if (args[1] == "christmas") {
      Christmas(hue, bulb_set);
    }
    else if (args[1] == "link") {
      Link(hue);
    }
    else {
      cerr << "Invalid command.\n\n";
      HelpExit();
    }
  }
  else if (args.size() == 2 && args[1].Is_f32()) {
    // Set each bulb to the given brightness value.
    for(u32 bulb: bulb_set) {
      hue.Bulb(bulb, args[1].Get_f32());
    }
  }

  else if (args.size() == 4
           && args[1].Is_f32() && args[2].Is_f32() && args[3].Is_f32()) {
    // Set each bulb to the given rgb value.
    for(u32 bulb: bulb_set) {
      hue.Bulb(bulb, args[1].Get_f32(), args[2].Get_f32(), args[3].Get_f32());
    }
  }

  else {
    cerr << "Unknown command.\n\n";
    HelpExit();
  }

  return 0;
}

